<?php
/**
 * @author      Elicus <hello@elicus.com>
 * @link        https://www.elicus.com/
 * @copyright   2021 Elicus Technologies Private Limited
 * @version     2.2.6
 */
class El_Testimonial_Form_Module extends ET_Builder_Module {

	public $slug       = 'el_md_testimonial_extended_form';
	public $vb_support = 'on';

	protected $module_credits = array(
		'module_uri' => 'https://diviextended.com/product/divi-testimonial-extended/',
		'author'     => 'Elicus',
		'uri'        => 'https://elicus.com/',
	);

	public function init() {
		$this->name             = esc_html__( 'Testimonial Form', 'divi-testimonial-extended' );
		$this->main_css_element = '%%order_class%%';
		add_filter( 'et_late_global_assets_list', array( $this, 'el_dte_late_assets' ), 10, 3 );
	}

	public function el_dte_late_assets( $assets_list, $assets_args, $et_dynamic_assets ) {
		if ( function_exists( 'et_get_dynamic_assets_path' ) && function_exists( 'et_is_cpt' ) ) {
			$cpt_suffix = et_is_cpt() ? '_cpt' : '';
			$assets_list['et_posts'] = array(
				'css' => $assets_args['assets_prefix'] . "/css/posts{$cpt_suffix}.css",
			);
			$assets_list['et_icons_all'] = array(
				'css' => $assets_args['assets_prefix'] . "/css/icons_all.css",
			);
			$assets_list['et_icons_fa'] = array(
				'css' => $assets_args['assets_prefix'] . "/css/icons_fa_all.css",
			);
			$assets_list['et_icons_social'] = array(
				'css' => $assets_args['assets_prefix'] . "/css/icons_base_social.css",
			);
		}
		return $assets_list;
	}

	public function get_settings_modal_toggles() {
		return array(
			'general' => array(
				'toggles' => array(
					'main_content' => array(
						'title'    => esc_html__( 'Content', 'divi-testimonial-extended' ),
						'priority' => 1,
					),
					'required_elements'     => array(
						'title'             => esc_html__( 'Required Elements', 'divi-testimonial-extended' ),
						'tabbed_subtoggles' => true,
						'sub_toggles'       => array(
							'author_name'         => array(
								'name' => 'Name',
							),
							'author_email'        => array(
								'name' => 'Email',
							),
							'testimonial_content' => array(
								'name' => 'Content',
							),
						),
						'priority'          => 2,
					),
					'non_required_elements' => array(
						'title'             => esc_html__( 'Non Required Elements', 'divi-testimonial-extended' ),
						'tabbed_subtoggles' => true,
						'sub_toggles'       => array(
							'testimonial_title'    => array(
								'name' => 'Title',
							),
							'testimonial_category' => array(
								'name' => 'Category',
							),
							'testimonial_rating'   => array(
								'name' => 'Rating',
							),
							'author_meta'          => array(
								'name' => 'Meta',
							),
						),
						'priority'          => 3,
					),
					'spam'                  => array(
						'title'    => esc_html__( 'Spam Protection', 'divi-testimonial-extended' ),
						'priority' => 4,
					),
				),
			),
			'advanced' => array(
				'toggles' => array(
					'text'          => array(
						'title' => esc_html__( 'Text', 'divi-testimonial-extended' ),
					),
					'captcha_settings' => array(
						'title' => esc_html__( 'Captcha Text', 'divi-testimonial-extended' ),
					),
					'notification_settings' => array(
						'title' => esc_html__( 'Notification Text', 'divi-testimonial-extended' ),
					),
					'input_fields' => array(
						'title' => esc_html__( 'Fields', 'divi-testimonial-extended' ),
					),
				),
			),
		);
	}

	public function get_advanced_fields_config() {
		return array(
			'fonts' => array(
				'notification_text' => array(
					'label'            => esc_html__( 'Notification', 'divi-testimonial-extended' ),
					'css'              => array(
						'main' => "{$this->main_css_element} .el-module-notification, {$this->main_css_element} .el-module-notification p",
					),
					'font_size' => array(
						'default'        => '16px',
						'range_settings' => array(
							'min'  => '8',
							'max'  => '100',
							'step' => '1',
						),
					),
					'text_orientation' => array(
						'default' => 'center',
					),
					'line_height' => array(
						'default'        => '1.7',
						'range_settings' => array(
							'min'  => '0.1',
							'max'  => '10',
							'step' => '0.1',
						),
					),
					'letter_spacing' => array(
						'default'        => '0',
						'range_settings' => array(
							'min'  => '0.1',
							'max'  => '10',
							'step' => '0.1',
						),
					),
					'toggle_slug' => 'notification_settings',
				),
				'captcha_text' => array(
					'label'            => esc_html__( 'Captcha', 'divi-testimonial-extended' ),
					'css'              => array(
						'main' => "{$this->main_css_element} .el-md-captcha-values",
					),
					'font_size'        => array(
						'default'        => '14px',
						'range_settings' => array(
							'min'  => '8',
							'max'  => '100',
							'step' => '1',
						),
					),
					'line_height' => array(
						'default'        => '1.7',
						'range_settings' => array(
							'min'  => '0.1',
							'max'  => '10',
							'step' => '0.1',
						),
					),
					'letter_spacing' => array(
						'default'        => '0',
						'range_settings' => array(
							'min'  => '0.1',
							'max'  => '10',
							'step' => '0.1',
						),
					),
					'hide_text_align' => true,
					'toggle_slug' => 'captcha_settings',
				),
			),
			'button'                => array(
				'submit' => array(
					'label'          => esc_html__( 'Button', 'divi-testimonial-extended' ),
					'css'            => array(
						'main' => '%%order_class%% .et_pb_button.el-testimonial-submit-button',
					),
					'margin_padding' => array(
						'css' => array(
							'margin'    => '%%order_class%% .et_pb_button.el-testimonial-submit-button',
							'padding'   => '%%order_class%% .et_pb_button.el-testimonial-submit-button',
							'important' => 'all',
						),
					),
					'no_rel_attr'    => true,
					'use_alignment'  => false,
					'box_shadow'     => false,
				),
			),
			'form_field' => array(
				'form_field' => array(
					'label'          => esc_html__( 'Fields', 'divi-testimonial-extended' ),
					'css'            => array(
						'main'                         => '%%order_class%% .module-form-field',
						'background_color'             => '%%order_class%% .module-form-field, %%order_class%% .radio-field-container',
						'background_color_hover'       => '%%order_class%% .module-form-field:hover, %%order_class%% .radio-field-container:hover',
						'focus_background_color'       => '%%order_class%% .module-form-field:focus, %%order_class%% .radio-field-container module-form-field.input[type="radio"]:active + label',
						'focus_background_color_hover' => '%%order_class%% .module-form-field:focus:hover, %%order_class%% .radio-field-container .module-form-field[type="radio"]:active:hover + label',
						'placeholder_focus'            => '%%order_class%% p .module-form-field:focus::-webkit-input-placeholder, %%order_class%% p .module-form-field:focus::-moz-placeholder, %%order_class%% p .module-form-field:focus:-ms-input-placeholder, %%order_class%% p textarea:focus::-webkit-input-placeholder, %%order_class%% p textarea:focus::-moz-placeholder, %%order_class%% p textarea:focus:-ms-input-placeholder',
						'padding'                      => '%%order_class%% .module-form-field',
						'margin'                       => '%%order_class%% .module-form-field',
						'form_text_color'              => '%%order_class%% .module-form-field, %%order_class%% .module-form-field[type="radio"] + label, %%order_class%% .el-module-form-field-container label',
						'form_text_color_hover'        => '%%order_class%% .module-form-field:hover, %%order_class%% .module-form-field[type="radio"]:hover + label, %%order_class%% .el-module-form-field-container label:hover',
						'focus_text_color'             => '%%order_class%% .module-form-field:focus',
						'focus_text_color_hover'       => '%%order_class%% .module-form-field:focus:hover, %%order_class%% .module-form-field[type="radio"]:active:hover + label, %%order_class%% .el-module-form-field-container label',
					),
					'box_shadow'     => false,
					'border_styles'  => false,
					'font_field'     => array(
						'font_size'      => array(
							'default'        => '14px',
							'range_settings' => array(
								'min'  => '1',
								'max'  => '100',
								'step' => '1',
							),
							'validate_unit'  => true,
						),
						'line_height'    => array(
							'default'        => '1em',
							'range_settings' => array(
								'min'  => '0.1',
								'max'  => '10',
								'step' => '0.1',
							),
						),
						'letter_spacing' => array(
							'default'        => '0px',
							'range_settings' => array(
								'min'  => '0',
								'max'  => '10',
								'step' => '1',
							),
							'validate_unit'  => true,
						),
						'css'            => array(
							'main'  => implode(
								', ',
								array(
									"{$this->main_css_element} .module-form-field",
									"{$this->main_css_element} .el-module-form-field-container label",
									"{$this->main_css_element} .module-form-field::placeholder",
									"{$this->main_css_element} .module-form-field::-webkit-input-placeholder",
									"{$this->main_css_element} .module-form-field::-moz-placeholder",
									"{$this->main_css_element} .module-form-field:-ms-input-placeholder",
									"{$this->main_css_element} .module-form-field[type=checkbox] + label",
									"{$this->main_css_element} .module-form-field[type=radio] + label",
								)
							),
							'hover' => array(
								"{$this->main_css_element} .module-form-field:hover",
								"{$this->main_css_element} .el-module-form-field-container label",
								"{$this->main_css_element} .module-form-field:hover::placeholder",
								"{$this->main_css_element} .module-form-field:hover::-webkit-input-placeholder",
								"{$this->main_css_element} .module-form-field:hover::-moz-placeholder",
								"{$this->main_css_element} .module-form-field:hover:-ms-input-placeholder",
								"{$this->main_css_element} .module-form-field[type=checkbox]:hover + label",
								"{$this->main_css_element} .module-form-field[type=radio]:hover + label",
							),
						),
					),
					'margin_padding' => array(
						'css' => array(
							'main'    => '%%order_class%% .module-form-field',
							'padding' => '%%order_class%% .module-form-field',
							'margin'  => '%%order_class%% .module-form-field',
						),
					),
					'tab_slug'     => 'advanced',
					'toggle_slug'  => 'input_fields',
				),
			),
			'custom_margin_padding' => array(
				'css' => array(
					'main'      => $this->main_css_element,
					'important' => 'all',
				),
			),
			'borders' => array(
				'default'       => array(
					'css' => array(
						'main'      => array(
							'border_radii'  => $this->main_css_element,
							'border_styles' => $this->main_css_element,
						),
						'important' => 'all',
					),
				),
				'input'         => array(
					'label_prefix' => 'Input Fields',
					'css'          => array(
						'main'      => array(
							'border_radii'  => "{$this->main_css_element} p input:not([type='file']), {$this->main_css_element} p textarea, {$this->main_css_element} p select",
							'border_styles' => "{$this->main_css_element} p input:not([type='file']), {$this->main_css_element} p textarea, {$this->main_css_element} p select",
						),
						'important' => 'all',
					),
					'defaults' => array(
						'border_radii' => 'on||||',
						'border_styles' => array(
							'width' => '1px',
							'color' => '#eee',
							'style' => 'solid',
						),
					),
					'tab_slug'     => 'advanced',
					'toggle_slug'  => 'input_fields',
				),
			),
			'filters'               => false,
			'max_width'             => array(),
			'text'                  => array(
				'css' => array(
					'text_orientation' => '%%order_class%% input, %%order_class%% textarea, %%order_class%% label',
					'text_shadow'      => '%%order_class%%, %%order_class%% input, %%order_class%% textarea, %%order_class%% label, %%order_class%% select',
				),
			),
		);
	}

	public function get_custom_css_fields_config() {
		return array(
			'contact_fields' => array(
				'label'    => esc_html__( 'Form Fields', 'divi-testimonial-extended' ),
				'selector' => '.module-input-field',
			),
			'text_field'     => array(
				'label'    => esc_html__( 'Testimonial Field', 'divi-testimonial-extended' ),
				'selector' => 'textarea.module-textarea-field',
			),
			'form_button'    => array(
				'label'    => esc_html__( 'Button', 'divi-testimonial-extended' ),
				'selector' => '.el-testimonial-submit-button',
			),
		);
	}

	public function get_fields() {
		$et_accent_color = et_builder_accent_color();

		return array(
			'success_message'        => array(
				'label'           => esc_html__( 'Success Message', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'main_content',
				'description'     => esc_html__( 'Type the message you want to display after successful form submission. Leave blank for default', 'divi-testimonial-extended' ),
			),
			'submit_button_text'     => array(
				'label'           => esc_html__( 'Submit Button Text', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'default'         => 'Submit',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'main_content',
				'description'     => esc_html__( 'Define the text of the form submit button.', 'divi-testimonial-extended' ),
			),
			'send_email_to_admin'    => array(
				'label'           => esc_html__( 'Send Email to Admin', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'main_content',
				'description'     => esc_html__( 'Here you can choose whether an email sent to admin.', 'divi-testimonial-extended' ),
			),
			'admin_email'            => array(
				'label'           => esc_html__( 'Admin Email', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'default'         => get_option( 'admin_email' ),
				'show_if'         => array(
					'send_email_to_admin' => 'on',
				),
				'tab_slug'        => 'general',
				'toggle_slug'     => 'main_content',
				'description'     => esc_html__( 'Define the email of the admin.', 'divi-testimonial-extended' ),
			),
			'name_label'             => array(
				'label'           => esc_html__( 'Author Name Field Label', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'default'         => 'Name',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'required_elements',
				'sub_toggle'      => 'author_name',
				'description'     => esc_html__( 'The label for your author name field.', 'divi-testimonial-extended' ),
			),
			'email_label'            => array(
				'label'           => esc_html__( 'Author Email Field Label', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'default'         => 'Email',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'required_elements',
				'sub_toggle'      => 'author_email',
				'description'     => esc_html__( 'The label for your author email field.', 'divi-testimonial-extended' ),
			),
			'send_email'             => array(
				'label'           => esc_html__( 'Send Confirmation Email to Author', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'default'         => 'on',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'required_elements',
				'sub_toggle'      => 'author_email',
				'description'     => esc_html__( 'Here you can choose whether a confirmation email sent to author.', 'divi-testimonial-extended' ),
			),
			'testimonial_label'      => array(
				'label'           => esc_html__( 'Content Field Label', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'default'         => 'Testimonial',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'required_elements',
				'sub_toggle'      => 'testimonial_content',
				'description'     => esc_html__( 'The label for your testimonial content field.', 'divi-testimonial-extended' ),
			),
			'title'                  => array(
				'label'           => esc_html__( 'Show Title', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'testimonial_title',
				'description'     => esc_html__( 'Here you can choose whether title field should appear in form.', 'divi-testimonial-extended' ),
			),
			'title_required'         => array(
				'label'           => esc_html__( 'Mark Title Field Required', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'show_if'         => array(
					'title' => 'on',
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'testimonial_title',
				'description'     => esc_html__( 'Here you can choose whether title field should be required in form.', 'divi-testimonial-extended' ),
			),
			'title_label'            => array(
				'label'           => esc_html__( 'Title Field Label', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'default'         => 'Testimonial Title',
				'show_if'         => array(
					'title' => 'on',
				),
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'testimonial_title',
				'description'     => esc_html__( 'The label for your testimonial title field.', 'divi-testimonial-extended' ),
			),
			'category'               => array(
				'label'           => esc_html__( 'Category', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'testimonial_category',
				'description'     => esc_html__( 'Here you can choose whether category field should appear in form.', 'divi-testimonial-extended' ),
			),
			'category_required'      => array(
				'label'           => esc_html__( 'Mark Category Field Required', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'show_if'         => array(
					'category' => 'on',
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'testimonial_category',
				'description'     => esc_html__( 'Here you can choose whether category field should appear in form.', 'divi-testimonial-extended' ),
			),
			'category_label'         => array(
				'label'           => esc_html__( 'Category Field Label', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'show_if'         => array(
					'category' => 'on',
				),
				'default'         => 'Select Category',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'testimonial_category',
				'description'     => esc_html__( 'The label for your category field.', 'divi-testimonial-extended' ),
			),
			'company'                => array(
				'label'           => esc_html__( 'Author Company', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'author_meta',
				'description'     => esc_html__( 'Here you can choose whether author company field should appear in form.', 'divi-testimonial-extended' ),
			),
			'company_required'       => array(
				'label'           => esc_html__( 'Mark Company Field Required', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'show_if'         => array(
					'company' => 'on',
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'author_meta',
				'description'     => esc_html__( 'Here you can choose whether company field should be required in form.', 'divi-testimonial-extended' ),
			),
			'company_label'          => array(
				'label'           => esc_html__( 'Author Company Field Label', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'show_if'         => array(
					'company' => 'on',
				),
				'default'         => 'Company',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'author_meta',
				'description'     => esc_html__( 'The label for your author company field.', 'divi-testimonial-extended' ),
			),
			'company_url'            => array(
				'label'           => esc_html__( 'Company URL', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'show_if'         => array(
					'company' => 'on',
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'author_meta',
				'description'     => esc_html__( 'Here you can choose whether company url field should appear in form.', 'divi-testimonial-extended' ),
			),
			'company_url_required'   => array(
				'label'           => esc_html__( 'Mark URL Field Required', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'show_if'         => array(
					'company'     => 'on',
					'company_url' => 'on',
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'author_meta',
				'description'     => esc_html__( 'Here you can choose whether url field should be required in form.', 'divi-testimonial-extended' ),
			),
			'url_label'              => array(
				'label'           => esc_html__( 'Company URL Field Label', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'show_if'         => array(
					'company'     => 'on',
					'company_url' => 'on',
				),
				'default'         => 'Company URL',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'author_meta',
				'description'     => esc_html__( 'The label for your author url field.', 'divi-testimonial-extended' ),
			),
			'designation'            => array(
				'label'           => esc_html__( 'Author Designation', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'author_meta',
				'description'     => esc_html__( 'Here you can choose whether author designation field should appear in form.', 'divi-testimonial-extended' ),
			),
			'designation_required'   => array(
				'label'           => esc_html__( 'Mark Designation Field Required', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'show_if'         => array(
					'designation' => 'on',
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'author_meta',
				'description'     => esc_html__( 'Here you can choose whether designation field should be required in form.', 'divi-testimonial-extended' ),
			),
			'designation_label'      => array(
				'label'           => esc_html__( 'Author Designation Field Label', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'show_if'         => array(
					'designation' => 'on',
				),
				'default'         => 'Designation',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'author_meta',
				'description'     => esc_html__( 'The label for your author designation field.', 'divi-testimonial-extended' ),
			),
			'image'                  => array(
				'label'           => esc_html__( 'Author Image', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'author_meta',
				'description'     => esc_html__( 'Here you can choose whether author image field should appear in form.', 'divi-testimonial-extended' ),
			),
			'image_required'         => array(
				'label'           => esc_html__( 'Mark Image Field Required', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'show_if'         => array(
					'image' => 'on',
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'author_meta',
				'description'     => esc_html__( 'Here you can choose whether author image field should be required in form.', 'divi-testimonial-extended' ),
			),
			'image_label'            => array(
				'label'           => esc_html__( 'Author Image Field Label', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'show_if'         => array(
					'image' => 'on',
				),
				'default'         => 'Image',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'author_meta',
				'description'     => esc_html__( 'The label for your author image field.', 'divi-testimonial-extended' ),
			),
			'rating'                 => array(
				'label'           => esc_html__( 'Rating', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'testimonial_rating',
				'description'     => esc_html__( 'Here you can choose whether rating field should appear in form.', 'divi-testimonial-extended' ),
			),
			'rating_required'        => array(
				'label'           => esc_html__( 'Mark Rating Field Required', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'show_if'         => array(
					'rating' => 'on',
				),
				'default'         => 'off',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'testimonial_rating',
				'description'     => esc_html__( 'Here you can choose whether rating field should be required in form.', 'divi-testimonial-extended' ),
			),
			'rating_label'           => array(
				'label'           => esc_html__( 'Rating Field Label', 'divi-testimonial-extended' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'show_if'         => array(
					'rating' => 'on',
				),
				'default'         => 'Rating',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'non_required_elements',
				'sub_toggle'      => 'testimonial_rating',
				'description'     => esc_html__( 'The label for your rating field.', 'divi-testimonial-extended' ),
			),
			'captcha'                => array(
				'label'           => esc_html__( 'Use Basic Captcha', 'divi-testimonial-extended' ),
				'type'            => 'yes_no_button',
				'option_category' => 'configuration',
				'options'         => array(
					'on'  => esc_html__( 'Yes', 'divi-testimonial-extended' ),
					'off' => esc_html__( 'No', 'divi-testimonial-extended' ),
				),
				'default'         => 'on',
				'tab_slug'        => 'general',
				'toggle_slug'     => 'spam',
				'description'     => esc_html__( 'Turn the captcha on or off using this option.', 'divi-testimonial-extended' ),
			),
			'__testimonial_category' => array(
				'type'                => 'computed',
				'computed_callback'   => array( 'El_Testimonial_Form_Module', 'el_get_testimonial_category' ),
				'computed_depends_on' => array(
					'category',
					'category_required',
				),
			),
		);
	}

	/**
	 * This function will return category for builder.
	 *
	 * @param array arguments to get testimonial data
	 * @return category array if testimonial category exist otherwise empty values.
	 * */
	public static function el_get_testimonial_category( $args = array(), $conditional_tags = array(), $current_page = array() ) {

		$defaults          = array(
			'category' => '',
		);
		$args              = wp_parse_args( $args, $defaults );
		$testimonial_terms = array();

		if ( 'on' === $args['category'] ) {
			$testimonial_terms = get_terms(
				array(
					'taxonomy'   => 'testimonial_category',
					'hide_empty' => false,
				)
			);
		}

		if ( ! empty( $testimonial_terms ) ) {
			return wp_json_encode( $testimonial_terms );
		} else {
			return '';
		}
	}

	public function render( $attrs, $content, $render_slug ) {

		$success_message      = esc_attr( $this->props['success_message'] );
		$submit_button_text   = esc_attr( $this->props['submit_button_text'] );
		$name_label           = esc_attr( $this->props['name_label'] );
		$title                = esc_attr( $this->props['title'] );
		$title_label          = esc_attr( $this->props['title_label'] );
		$title_required       = esc_attr( $this->props['title_required'] );
		$category             = esc_attr( $this->props['category'] );
		$category_label       = esc_attr( $this->props['category_label'] );
		$category_required    = esc_attr( $this->props['category_required'] );
		$email_label          = esc_attr( $this->props['email_label'] );
		$send_email           = esc_attr( $this->props['send_email'] );
		$company              = esc_attr( $this->props['company'] );
		$company_label        = esc_attr( $this->props['company_label'] );
		$company_required     = esc_attr( $this->props['company_required'] );
		$company_url          = esc_attr( $this->props['company_url'] );
		$url_label            = esc_attr( $this->props['url_label'] );
		$company_url_required = esc_attr( $this->props['company_url_required'] );
		$designation          = esc_attr( $this->props['designation'] );
		$designation_label    = esc_attr( $this->props['designation_label'] );
		$designation_required = esc_attr( $this->props['designation_required'] );
		$image                = esc_attr( $this->props['image'] );
		$image_label          = esc_attr( $this->props['image_label'] );
		$image_required       = esc_attr( $this->props['image_required'] );
		$rating               = esc_attr( $this->props['rating'] );
		$rating_label         = esc_attr( $this->props['rating_label'] );
		$rating_required      = esc_attr( $this->props['rating_required'] );
		$testimonial_label    = esc_attr( $this->props['testimonial_label'] );
		$captcha              = esc_attr( $this->props['captcha'] );

		wp_enqueue_script( 'el-testimonial-form-script' );

		if ( '' === $submit_button_text ) {
			$submit_button_text = esc_html__( 'Submit', 'divi-testimonial-extended' );
		} else {
			$submit_button_text = sprintf(
				esc_html__( '%s', 'divi-testimonial-extended' ),
				esc_html( $submit_button_text )
			);
		}
		$submit_button   = $this->props['custom_submit'];
		$submit_use_icon = $this->props['submit_use_icon'];
		$submit_icon     = $this->props['submit_icon'];
		$first_digit     = wp_rand( 1, 20 );
		$second_digit    = wp_rand( 1, 20 );
		$captcha_actual  = (int) esc_attr( $first_digit ) + (int) esc_attr( $second_digit );
		$captcha_field   = '';

		$host 			= site_url(); //isset( $_SERVER['HTTP_HOST'] ) ? wp_unslash( $_SERVER['HTTP_HOST'] ) : '';
		$request_uri 	= isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
		$page_url    	= $host . $request_uri;

		$flag         = 0;
		$order_class  = $this->get_module_order_class( 'el_md_testimonial_extended_form' );
		$order_number = str_replace( '_', '', str_replace( 'el_md_testimonial_extended_form', '', $order_class ) );

		if ( isset( $_POST['form-number'] ) && esc_attr( $order_number ) === sanitize_text_field( wp_unslash( $_POST['form-number'] ) ) ) {
			if ( isset( $_POST['parameters'], $_POST['values'], $_POST['nonce-el-md-testimonial-extended-form'] ) ) {
				if ( wp_verify_nonce( sanitize_key( wp_unslash( $_POST['nonce-el-md-testimonial-extended-form'] ) ), 'el-md-testimonial-extended-form' ) ) {
					// phpcs:ignore ET.Sniffs.ValidatedSanitizedInput.InputNotSanitized
					if ( 'on' === $captcha && isset( $_POST['captcha'] ) ) {
						if ( ! empty( $_POST['captcha'] ) ) {
							$flag = 1;
						} else {
							$form_message = esc_html__( 'Invalid Captcha Values. Please refresh the page and try again.', 'divi-testimonial-extended' );
						}
					} else {
						if ( 'off' === $captcha ) {
							$flag = 1;
						} else {
							$form_message = esc_html__( 'Invalid submission. Please refresh the page and try again.', 'divi-testimonial-extended' );
						}
					}

					if ( 1 === $flag ) {
						// phpcs:disable ET.ValidatedSanitizedInput.InputNotSanitized
						$status = $this->el_add_testimonial(
							wp_unslash( et_()->array_get_sanitized( $_POST, 'parameters' ) ),
							wp_unslash( et_core_intentionally_unescaped( $_POST['values'], 'html' ) ) // phpcs:ignore ET.Sniffs.ValidatedSanitizedInput.InputNotSanitized
						);
						// phpcs:enable ET.ValidatedSanitizedInput.InputNotSanitized
						$form_message = ( 1 === $status ) ?
										( '' !== $success_message ?
											sprintf( esc_html__( '%s', 'divi-testimonial-extended' ), esc_html( $success_message ) ) :
											esc_html__( 'Thank You.', 'divi-testimonial-extended' ) ) :
										$status;
					}
				} else {
					$form_message = esc_html__( 'Cheatin\' uh?', 'divi-testimonial-extended' );
				}
				$results = '<div class="el-module-notification">' . esc_html( $form_message ) . '</div>';
				return et_core_intentionally_unescaped( $results, 'html' );
			}
		}

		if ( 'on' === $category ) {
			$testimonial_terms = get_terms(
				array(
					'taxonomy'   => 'testimonial_category',
					'hide_empty' => false,
				)
			);
		}

		$fields = sprintf(
			'<p class="el-module-form-field-container">
						<input type="text" name="name" class="module-form-field module-input-field" placeholder="%1$s*" data-required="required" />
					</p>
					<p class="el-module-form-field-container">
						<input type="email" name="email" class="module-form-field module-input-field module-email-field" placeholder="%2$s*" data-required="required" />
					</p>',
			esc_attr( $name_label ),
			esc_attr( $email_label )
		);

		$text_fields = array(
			'title_label'       => 'title',
			'company_label'     => 'company',
			'url_label'         => 'company_url',
			'designation_label' => 'designation',
		);

		foreach ( $text_fields as $label => $field ) {
			$required = 'on' === ${$field . '_required'} ? ' data-required=required' : '';
			$star     = 'on' === ${$field . '_required'} ? '*' : '';
			if ( 'on' === ${$field} && 'company_url' !== $field ) {
				$fields .= sprintf(
					'<p class="el-module-form-field-container">
						<input type="text" name="%1$s" class="module-form-field module-input-field" placeholder="%2$s%3$s" %4$s />
					</p>',
					esc_attr( $field ),
					esc_attr( ${$label} ),
					esc_attr( $star ),
					esc_attr( $required )
				);
			} elseif ( 'on' === ${$field} && 'on' === $company && 'company_url' === $field ) {
				$fields .= sprintf(
					'<p class="el-module-form-field-container">
						<input type="url" name="%1$s" class="module-form-field module-input-field" placeholder="%2$s%3$s" %4$s />
					</p>',
					esc_attr( $field ),
					esc_attr( ${$label} ),
					esc_attr( $star ),
					esc_attr( $required )
				);
			}
		}

		if ( 'on' === $category && ! is_wp_error( $testimonial_terms ) && ! empty( $testimonial_terms ) ) {
			$required = 'on' === $category_required ? ' data-required=required' : '';
			$star     = 'on' === $category_required ? '*' : '';
			$options  = sprintf(
				'<option value="">%1$s%2$s</option>',
				esc_html( $category_label ),
				esc_html( $star )
			);
			foreach ( $testimonial_terms as $term ) {
				$options .= sprintf(
					'<option value="%1$s">%2$s</option>',
					esc_attr( $term->term_id ),
					esc_html( $term->name )
				);
			}
			$fields .= sprintf(
				'<p class="el-module-form-field-container">
					<select class="module-form-field module-input-field" name="category" *1$s>
					%2$s
					</select>
				</p>',
				esc_attr( $required ),
				et_core_intentionally_unescaped( $options, 'html' )
			);
		}

		if ( 'on' === $rating ) {
			$required = 'on' === $rating_required ? ' data-required=required' : '';
			$star     = 'on' === $rating_required ? '*' : '';
			$options  = '';
			for ( $i = 1; $i <= 5; $i++ ) {
				$options .= sprintf(
					'<input type="radio" name="rating" class="module-form-field module-radio-field" value="%1$s" />
					<label>%2$s</label>',
					esc_attr( $i ),
					esc_html( $i )
				);
			}
			$fields .= sprintf(
				'<p class="el-module-form-field-container">
					<label>%1$s%2$s</label>
					<span class="radio-field-container">
					%3$s
					<input type="hidden" name="rating" class="module-form-field module-radio-value-field module-input-field" %4$s />
					</span>
				</p>',
				esc_html( $rating_label ),
				esc_html( $star ),
				et_core_intentionally_unescaped( $options, 'html' ),
				esc_attr( $required )
			);
		}

		$fields .= sprintf(
			'<p class="el-module-form-field-container full-width-field">
				<textarea name="testimonial" class="module-form-field module-input-field module-textarea-field" placeholder="%1$s*" data-required="required"></textarea>
			</p>',
			esc_attr( $testimonial_label )
		);

		if ( 'on' === $image ) {
			$required = 'on' === $image_required ? ' data-required=required' : '';
			$star     = 'on' === $image_required ? '*' : '';
			$fields  .= sprintf(
				'<p class="el-module-form-field-container full-width-field">
					<label>%1$s%2$s</label>
					<input type="file" class="module-form-field module-input-field module-file-field" %3$s />
				</p>',
				esc_html( $image_label ),
				esc_html( $star ),
				esc_attr( $required )
			);
		}

		if ( 'on' === $captcha ) {
			$captcha_field = sprintf(
				'<span class="el-md-captcha-values">%1$s + %2$s =</span>
				<input class="module-form-field module-form-captcha" type="text" name="captcha" size="2" data-first-digit="%1$s" data-second-digit="%2$s" data-required="required" autocomplete="disabled" />',
				esc_attr( $first_digit ),
				esc_attr( $second_digit )
			);
		}

		$fields .= sprintf(
			'<p class="el-module-form-field-container full-width-field el-md-button-container">
				%2$s
				<button type="submit" class="et_pb_button module-submit-button el-testimonial-submit-button%4$s"%3$s>%1$s</button>
			</p>',
			$submit_button_text,
			et_core_intentionally_unescaped( $captcha_field, 'html' ),
			(
				'on' === $submit_use_icon && '' !== $submit_icon && 'on' === $submit_button ?
				sprintf(
					' data-icon="%1$s"',
					esc_attr( et_pb_process_font_icon( $submit_icon ) )
				) :
				''
			),
			(
				'on' === $submit_use_icon && '' !== $submit_icon && 'on' === $submit_button ?
				' et_pb_custom_button_icon' :
				''
			)
		);

		if ( 'on' === $send_email ) {
			$fields .= '<input type="hidden" name="send_email" class="module-form-field module-input-field" value="on" />';
		}

		$fields .= wp_nonce_field( 'el-md-testimonial-extended-form', 'nonce-el-md-testimonial-extended-form', true, false );

		$form = sprintf(
			'<div class="el-divi-testimonial-extended-form">
				<form class="el_testimonial_form_wrapper" method="post" action="%1$s" data-form-number="%3$s">
					%2$s
				</form>
			</div>',
			esc_url( $page_url ),
			et_core_intentionally_unescaped( $fields, 'html' ),
			esc_attr( $order_number )
		);

		return et_core_intentionally_unescaped( $form, 'html' );
	}

	/**
	 * This function will upload testimonial images into media library.
	 *
	 * @return int
	 * */
	public function el_upload_file_to_media_library() {

		if ( isset( $_FILES['file']['name'] ) ) {

			$uploaded_file_name = sanitize_file_name( wp_unslash( $_FILES['file']['name'] ) );
			$valid_formats      = array( 'jpg', 'png', 'jpeg' );
			$wp_upload_dir      = wp_upload_dir();
			$path               = $wp_upload_dir['path'] . '/';
			$mimes              = array(
				'image/jpeg',
				'image/gif',
				'image/png',
				'image/bmp',
			);

			$check_file = wp_check_filetype( $uploaded_file_name );

			if ( in_array( $check_file['type'], $mimes, true ) ) {
				$upload_overrides = array(
					'test_form' => false,
				);

				// phpcs:ignore ET.Sniffs.ValidatedSanitizedInput.InputNotSanitized
				$unslashed_file = wp_unslash( $_FILES['file'] );

				if ( isset( $unslashed_file ) && '' !== $unslashed_file ) {
					$movefile = wp_handle_upload( $unslashed_file, $upload_overrides );
				}

				if ( isset( $movefile ) && isset( $movefile['url'] ) ) {

					$new_name  = wp_basename( esc_url_raw( $movefile['url'] ) );
					$file_path = esc_url_raw( $path . $new_name );

					$attachment = array(
						'guid'           => esc_url_raw( $movefile['url'] ),
						'post_mime_type' => $check_file['type'],
						'post_title'     => preg_replace( '/\.[^.]+$/', '', $new_name ),
						'post_content'   => '',
						'post_status'    => 'inherit',
					);
					$attach_id  = wp_insert_attachment( $attachment, $file_path );
					require_once ABSPATH . 'wp-admin/includes/image.php';
					$attach_data = wp_generate_attachment_metadata( $attach_id, $file_path );
					wp_update_attachment_metadata( $attach_id, $attach_data );
					return (int) $attach_id;
				}
			} else {
				return 0;
			}
		} else {
			return 0;
		}
	}

	/**
	 * This function sends email to user who have submitted the testimonial.
	 *
	 * @param string $to whom to send the email.
	 * @param string $subject subject line of email.
	 * @param string $message content of email.
	 * @param array  $headers email header.
	 * @return void
	 * */
	public function el_send_email( $to, $subject, $message, $headers ) {
		wp_mail( $to, $subject, $message, $headers );
	}

	/**
	 * This function will create testimonial custom post.
	 *
	 * @param array $param custom field names.
	 * @param array $val custom fields values.
	 *
	 * @return 1 if created otherwise an error message.
	 * */
	public function el_add_testimonial( $param, $val ) {
		$parameters = explode( '##', esc_attr( $param ) );
		$values     = explode( '##', $val );
		$data       = array();
		$loop       = count( $parameters );
		$email      = '';

		for ( $i = 0; $i < $loop; $i++ ) {
			if ( 'testimonial' === $parameters[ $i ] ) {
				$data[ $parameters[ $i ] ] = nl2br( $values[ $i ] );
			} else {
				$data[ $parameters[ $i ] ] = esc_attr( $values[ $i ] );
			}
		}

		if ( isset( $data['title'] ) && '' !== sanitize_text_field( $data['title'] ) ) {
			$title   = ucfirst( sanitize_text_field( $data['title'] ) );
		} else {
			$title 	= 'Auto Draft';
		}

		if ( isset( $data['testimonial'] ) && '' !== sanitize_textarea_field( $data['testimonial'] ) ) {
			$content = wp_kses_post( wpautop( $data['testimonial'] ) );
		} else {
			$content = '';
		}

		if ( isset( $data['category'] ) ) {
			$category = array();
			array_push( $category, esc_attr( $data['category'] ) );
			unset( $data['category'] );
		}
		if ( isset( $data['send_email'] ) ) {
			$email = 'on' === $data['send_email'] ? 'yes' : '';
			unset( $data['send_email'] );
		}

		unset( $data['title'] );
		unset( $data['testimonial'] );

		$attach_id = $this->el_upload_file_to_media_library();

		$post_id = wp_insert_post( array(
			'post_type'    => 'testimonials',
			'post_title'   => et_core_esc_previously( $title ),
			'post_content' => et_core_esc_previously( $content ),
			'post_status'  => 'draft',
		) );

		if ( 0 !== $post_id ) {
			if ( 0 !== $attach_id ) {
				set_post_thumbnail( $post_id, $attach_id );
			}
			if ( isset( $category ) && ! empty( $category ) ) {
				wp_set_post_terms( $post_id, $category, 'testimonial_category' );
			}

			foreach ( $data as $key => $value ) {
				$value = sanitize_text_field( $value );
				$key   = 'testimonial_meta_box-' . $key;
				$key   = sanitize_key( $key );
				update_post_meta( $post_id, $key, $value );
			}
			if ( 'on' === $this->props['send_email_to_admin'] ) {
				$admin_email = sanitize_email( $this->props['admin_email'] );
				if ( '' === $admin_email ) {
					$admin_email = sanitize_email( get_option( 'admin_email' ) );
				}
				$blogname = esc_html( get_option( 'blogname' ) );
				$subject  = esc_html__( 'New Testimonial On ', 'divi-testimonial-extended' ) . $blogname;
				$message  = esc_html__( 'You have received a new testimonial from ', 'divi-testimonial-extended' ) . ucfirst( esc_html( $data['name'] ) ) . ',<br/>' . esc_html__( 'To approve visit this url ', 'divi-testimonial-extended' ) . esc_html( admin_url( 'post.php?post=' . $post_id . '&action=edit' ) ) . '<br/><br/>' . esc_html__( 'Regards,', 'divi-testimonial-extended' ) . '<br/>' . $blogname;
				$headers  = array(
					'From: ' . $data['email'],
					'Content-Type: text/html; charset=UTF-8',
				);
				$this->el_send_email( $admin_email, $subject, $message, $headers );
			}

			if ( 'yes' === $email ) {
				$blogname    = esc_html( get_option( 'blogname' ) );
				$subject     = esc_html__( 'Testimonial Submitted on ', 'divi-testimonial-extended' ) . $blogname;
				$message     = esc_html__( 'Hi ', 'divi-testimonial-extended' ) . ucfirst( esc_html( $data['name'] ) ) . ',<br/>' . esc_html__( 'Thank you for your testimonial.', 'divi-testimonial-extended' ) . '<br/><br/>' . esc_html__( 'Regards,', 'divi-testimonial-extended' ) . '<br/>' . $blogname;
				$admin_email = isset( $admin_email ) ? $admin_email : sanitize_email( get_option( 'admin_email' ) );
				$headers     = array(
					'From: ' . $admin_email,
					'Content-Type: text/html; charset=UTF-8',
				);
				$this->el_send_email( $data['email'], $subject, $message, $headers );
			}

			return 1;
		} else {
			return esc_html__( 'Invalid submission. Please refresh the page and try again.', 'divi-testimonial-extended' );
		}
	}

}

new El_Testimonial_Form_Module();